***************************************************************
* This Stata dofile produces the results in: 
* Andrew Leigh, "Does Raising the Minimum Wage Help the Poor?" (2007) Economic Record 83(263): 432-445.
* Please cite this paper if you use the code that follow.
* Questions to andrew_leigh@ksg02.harvard.edu.
*
* The data are from the 1994/95 to 2002/03 SIHC CURFs
* See www.abs.gov.au for details on obtaining these.
***************************************************************

clear
set mem 100m
set more off
version 9

* Ends with h=household
* Ends with u=income unit
* Ends with p=person
* hrswkmcp- hours, main job;  hrswkacp - hours, all jobs
* 1994-99: lfstfcp=empstat. 2000: empstat=empstat

* Setup: 1994-2000
for any 94 95 96 97 \ num 1994/1997: cd "<Active directory>" \ use idsX, clear \ keep ftpt* cob statecu depkidsu kid1524u kid2124u kid1920u iutype personsu studstcp randomid iuno pno hhidu statecu qtritrwu iwsucu inctotcu inctotpu itaxcu itaxpu wtunit sexp agecp mstatcp cob occcp indcp hqualcp hrswkmcp hrswkacp lfstfcp empd6m1-empd6m7 iwsucp inctotcp itaxcp itaxpu wtpsn \ ren lfstfcp empstat \ gen pid=(hhidu*100)+(iuno*10)+pno \ gen year=Y \ sort pid year \ cd "C:\Users\Andrew\My publications\Min Wage - May Discontinuity\SIHC\" \ save tempY, replace
for any 99 \ num 1999: cd "<Active directory>" \ use idsX, clear \ keep ftpt* cob statecu depkidsu kid1524u kid2124u kid1920u iutype personsu studstcp abshid absfid absiid abspid statecu qtritrwu iwsucu inctotcu inctotpu itaxcu itaxpu wtunit sexp agecp mstatcp cob occcp indcp hqualcp hrswkmcp hrswkacp lfstfcp empd6m1-empd6m7 iwsucp inctotcp itaxcp itaxpu wtpsn \ ren lfstfcp empstat \ gen randomid=substr(abshid,6,8) \ destring randomid, force replace \ gen iuno=(absfid*10)+absiid \ gen pno=abspid \ gen hhidu=randomid \ drop abshid absfid absiid abspid \ gen pid=(hhidu*100)+(iuno*10)+pno \ gen year=Y \ sort pid year \ cd "C:\Users\Andrew\My publications\Min Wage - May Discontinuity\SIHC\" \ save tempY, replace
for any 00 \ num 2000: cd "<Active directory>" \ use idsX, clear \ keep ftpt* cob statecu depkidsu kid1524u kid2124u kid1920u iutype personsu studstcp randomid iuno pno iwsuch inctotch hhidu statecu qtritrwu iwsucu inctotcu inctotpu itaxcu itaxpu wtunit sexp agecp mstatcp cob occcp indcp hqualcp hrswkmcp hrswkacp empstat empd6m1-empd6m7 iwsucp inctotcp itaxcp itaxpu wtpsn inctotph wthh \ gen hours2000=hrswkacp \ recode hrswkacp 1=2 2=4.5 3=6.5 4=8.5 5=10.5 6=12.5 7=14.5 8=16.5 9=18.5 10=20.5 11=22.5 12=24.5 13=26.5 14=28.5 15=30.5 16=33 17=35 18=36.5 19=38.5 20=40 21=41.5 22=43.5 23=45.5 24=48 25=50 \ gen pid=(hhidu*100)+(iuno*10)+pno \ gen year=Y \ sort pid year \ cd "C:\Users\Andrew\My publications\Min Wage - May Discontinuity\SIHC\" \ save tempY, replace

* Setup: 2002
* Note that 2002 has no quarter of interview, so we set the quarter for all observations to 200208 (Aug 2002).
cd "<Active directory>" 
use ids02hh, clear
sort abshid
save, replace
use ids02iu, clear
sort absiid absfid abshid
save, replace
use ids02per, clear
sort absiid absfid abshid
merge absiid absfid abshid using ids02iu
tab _merge
drop _merge
sort abshid
merge abshid using ids02hh
tab _merge
drop _merge
ren depkidbc depkidsu
ren kid1524b kid1524u
ren statehbc statecu
ren cobl cob
ren prsnsubc personsu
gen randomid=substr(abshid,6,8) 
for var kid* ind* occ* hqual* age* dep* persons* absfid abspid absiid randomid hrs*: destring X, force replace
gen iuno=(absfid*10)+absiid
gen pno=abspid 
gen hhidu=randomid 
drop abshid absfid absiid abspid 
for any age ind hqual hrswkm hrswka: ren Xbc Xcp
ren occcbc occcp
gen kidtemp2002=kid0t2bc+kid3t4bc+kid5t9bc+kid1014b
keep kidtemp2002 ftpt* cob statecu depkidsu kid1524u iutype personsu studstcp randomid iuno pno itgcbcu iwsuch inctotch hhidu iwsucu inctotcu inctotpu itaxcu itaxpu wtunit sexp agecp mstatcp occcp indcp hqualcp hrswkmcp hrswkacp empstat empd6m1-empd6m7 iwsucp inctotcp itaxcp itaxpu wtpsn inctotph wthh 
gen hours2000=hrswkacp
recode hrswkacp 1=2 2=4.5 3=6.5 4=8.5 5=10.5 6=12.5 7=14.5 8=16.5 9=18.5 10=20.5 11=22.5 12=24.5 13=26.5 14=28.5 15=30.5 16=33 17=35 18=36.5 19=38.5 20=40 21=41.5 22=43.5 23=45.5 24=48 25=50
replace wtunit=int(wtunit)
gen pid=(hhidu*100)+(iuno*10)+pno 
gen year=2002
sort pid year
cd "<Active directory>" 
save temp2002, replace

***

cd "<Active directory>" 
use temp1994, clear
for num 1995/1997 1999/2000 2002: sort pid year \ merge pid year using tempX \ drop _merge
gen inctotpu2=inctotpu
recode inctotpu2 min/1=1
* Estimating ginis of unequivalised weekly income unit income
for num 1994 1995/1997 1999/2000 2002: inequal inctotcu [fw=wtunit] if year==X & pno==1
gen hours=hrswkacp 
recode hours 0=. 
gen hwage=iwsucp/hours
la var hwage "Hourly wage"
gen atr=itaxcp/inctotcp
recode atr min/0=0
gen hwage_aftertax=hwage*(1-atr)

ren cob osborn
recode osborn 1=0 .=. *=1
la var osborn "Overseas born"

* Recode quarter to the month that is in the middle of that quarter, eg. March quarter runs Jan-March, so we code it as Feb.
recode qtr 1=8 2=11 3=2 4=5
label define qtrlab 2 "Jan-Mar" 5 "Apr-Jun" 8 "Jul-Sep" 11 "Oct-Dec"
label values qtr qtrlab
la var qtr "Quarter of interview"

gen survey=year
replace year=year+1 if qtr<=5
gen yearmonth=(year*100)+qtr
replace yearmonth=200208 if year==2002

gen mw=.
gen mw_change=.
gen mw_fwd=.
la var mw "Minimum wage (C14)"
la var mw_change "Time before/after MW change"
* Note that MW date is the later of the date of decision and the operative date. 
* In 1994 and 1995, MW didn't take effect until 12/93, 11/94 respectively. In 1995, decision handed down on 10/95, but backdated to 4/95.
for num 325.4 333.4 333.4 341.4 349.4 359.4 373.4 385.4 400.4 413.4 431.4 448.4 467.4 \ num 199109 199312 199411 199510 199605 199705 199805 199905 200005 200105 200205 200305 200405: replace mw=X/38 if yearmonth>=Y \ replace mw_change=yearmonth-Y if mw_change>12 | mw_change<-12 \ replace mw_fwd=X/38 if yearmonth+9>=Y
for any cpi salwages gdp u u_lag month: gen X=.
for num 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 200408 200411 \ num 114.60 115.30 116.10 116.70 117.80 119.00 120.50 121.50 121.90 122.80 123.30 124.20 125.10 125.40 125.20 126.10 126.50 127.20 127.52 128.15 128.04 128.57 129.68 130.26 131.22 132.12 133.15 133.56 134.98 136.10 136.51 137.73 138.95 139.96 140.88 141.90 143.73 143.73 144.54 145.25 146.58 147.29 147.90 149.02: replace cpi=Y/149.02 if yearmonth>=X
for num 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 \ num 48223 50898 49517 51503 51059 53151 53230 55728 54369 56156 57142 60368 57420 59714 59723 62583 59905 62367 63959 66919 63707 66031 66625 70272 68020 70049 71476 74256 71715 73949 75269 78106 75024 77853 79384 82765 79604 82827 83800 87372 84044 87422: replace salwages=Y if yearmonth==X
for num 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 \ num 109022 113392 115601 122776 113776 119194 121275 132036 122234 127283 128589 139407 127230 134660 135993 147593 135499 142144 144279 154850 143855 148934 150975 161475 153900 159686 164673 173461 163273 169713 172409 186349 174600 181012 184469 196508 184663 190531 195491 211038 198387 205960: replace gdp=Y if yearmonth==X
for num 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 \ num 1(3)126: replace month=Y if yearmonth==X
for num 199102 199105 199108 199111 199202 199205 199208 199211 199302 199305 199308 199311 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 200408 200411 \ num 8.4 9.2 9.6 9.9 10.1 10.5 10.6 10.8 10.7 10.6 10.6 10.5 10.1 9.7 9.2 8.8 8.5 8.1 8.1 8.1 8.2 8.1 8.3 8.3 8.4 8.3 8.2 8.0 7.9 7.7 7.8 7.5 7.2 6.9 6.9 6.6 6.5 6.3 6.1 6.2 6.5 6.9 6.9 7.0 6.7 6.4 6.3 6.2 6.2 6.2 6.0 5.8 5.7 5.6 5.6 5.2: replace u=Y if yearmonth>=X 
for num 199105 199108 199111 199202 199205 199208 199211 199302 199305 199308 199311 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 200408 200411 \ num 8.4 9.2 9.6 9.9 10.1 10.5 10.6 10.8 10.7 10.6 10.6 10.5 10.1 9.7 9.2 8.8 8.5 8.1 8.1 8.1 8.2 8.1 8.3 8.3 8.4 8.3 8.2 8.0 7.9 7.7 7.8 7.5 7.2 6.9 6.9 6.6 6.5 6.3 6.1 6.2 6.5 6.9 6.9 7.0 6.7 6.4 6.3 6.2 6.2 6.2 6.0 5.8 5.7 5.6 5.6: replace u_lag=Y if yearmonth>=X 
gen gdp_sal=ln(gdp-salwages)
la var gdp_sal "Log GDP minus salary and wages component"
la var u "Unemployment rate"

*******************
gen temp_ym=.
for num 1994/1997 1999/2000 2002: replace temp_ym=X*100+12 if survey==X
for num 325.4 333.4 333.4 341.4 349.4 359.4 373.4 385.4 400.4 413.4 431.4 448.4 467.4 \ num 199109 199312 199411 199510 199605 199705 199805 199905 200005 200105 200205 200305 200405: replace mw=X/38 if temp_ym>=Y 
drop temp_ym

replace gdp=ln(gdp)
gen mw_ln=ln(mw)
sum age if hwage/mw<.5,d
* Dropping outliers
drop if (hwage/cpi>100 | hwage/mw<.5) & hwage~=.

gen hwage_ln=ln(hwage)
gen mw_real=ln(mw/cpi)
gen hwage_real=ln(hwage/cpi)
gen lnhours=ln(hours)
la var hours "Hours per week, workers only"
gen employed=empstat
* Self-employed/business owner/paid in kind are codes 3/6 in 1994-98 surveys
recode employed 1/6=1 7/8=0 *=. if yearmonth<=199805
recode employed 1/5=1 6/7=0 *=. if survey==1999
recode employed 2/3=0 if survey>=2000 & survey~=.
bysort survey: sum employed

* Defining minimum wage workers and subminimum wage workers
gen sub_worker=1 if hwage<mw & mw~=. & hwage~=.
gen min_worker=1 if hwage>=mw & hwage<=1.2*mw & mw~=. & hwage~=.
for any sub min: recode X_worker .=0 if mw~=. & hwage~=. \ la var X_worker "X worker"

ren sexp female
recode female 2=1 1=0
ren agecp age
recode age 1=15 2=16 3=17 4=18 5=19 6=20 7=21 8=22 9=23 10=24 11=27 12=32 13=37 14=42 15=47 16=52 17=55 18=56 19=57 20=58 21=59 22=60 23=61 24=62 25=63 26=64 27=67 28=72 29=75 if survey<=2000
recode age 1=15 2=16 3=17 4=18 5=19 6=20 7=21 8=22 9=23 10=24 11=27 12=32 13=37 14=42 15=47 16=52 17=55 18=56 19=57 20=58 21=59 22=60 23=61 24=62 25=63 26=64 27=67 28=72 29=77 30=80 if survey==2002
gen agesq=(age^2)/1000
ren mstatcp married
recode married 1=1 2/3=0

gen kidsu15=depkidsu-kid1524u
replace kidsu15=kidtemp2002 if year==2002
drop kidtemp2002
gen haskids15=kidsu15
recode haskids15 .=. 1/max=1
gen school=studstcp
recode school 2=. 3=0 4=0
la var school "1=in school, 0=PT or no study (FT study=missing)"
gen studying=studstcp
recode studying 1/3=1 4=0
la var studying "Studying: school, FT, or PT"

recode hqualcp 1=1 2/4=2 5/8=3 9=4 if survey<=1997
label define edyears 1 "still at school" 2 "higher/bachelor degree" 3 "other post school qualifications" 4 "no qualifications"
label values hqualcp edyears
tab hqualcp, gen(educ)
for var iwsucp iwsucu inctotcu: recode X min/1=1

gen inc_salary=(iwsucp)
replace inc_salary=0 if iwsucp==. & employed==0
gen inc_salarypt=((inc_salary)*(1-atr))
gen inc_all=(inctotpu/(personsu^.5))
gen inc_allpt=((inctotpu-itaxpu)/(personsu^.5))
for any inc_salary inc_salarypt inc_all inc_allpt: recode X min/1=1 \ replace X=ln(X)

gen temp1=(inctotpu-itaxpu)/(personsu^.5)
for any inc inc55 inc25 ince: gen pctile_X=.
for num 1994/1997 1999/2000 2002: xtile tempX=temp1 [w=wtunit] if survey==X, nq(100) \ replace pctile_inc=tempX if survey==X \ drop tempX
for num 1994/1997 1999/2000 2002: xtile tempX=temp1 [w=wtunit] if survey==X & age<=55, nq(100) \ replace pctile_inc55=tempX if survey==X \ drop tempX
for num 1994/1997 1999/2000 2002: xtile tempX=temp1 [w=wtunit] if survey==X & age<=55 & age>=25, nq(100) \ replace pctile_inc25=tempX if survey==X \ drop tempX
bysort hhidu iuno survey: egen emp_any=max(employed)
for num 1994/1997 1999/2000 2002: xtile tempX=temp1 [w=wtunit] if survey==X & emp_any==1, nq(100) \ replace pctile_ince=tempX if survey==X \ drop tempX
drop temp1

for any hw hw55 hw25 hwe: gen pctile_X=.
for num 1994/1997 1999/2000 2002: xtile tempX=hwage [w=wtpsn] if survey==X, nq(100) \ replace pctile_hw=tempX if survey==X \ drop tempX
for num 1994/1997 1999/2000 2002: xtile tempX=hwage [w=wtpsn] if survey==X & age<=55, nq(100) \ replace pctile_hw55=tempX if survey==X \ drop tempX
for num 1994/1997 1999/2000 2002: xtile tempX=hwage [w=wtpsn] if survey==X & age<=55 & age>=25, nq(100) \ replace pctile_hw25=tempX if survey==X \ drop tempX
for num 1994/1997 1999/2000 2002: xtile tempX=hwage [w=wtpsn] if survey==X & emp_any==1, nq(100) \ replace pctile_hwe=tempX if survey==X \ drop tempX

* How correlated are equivalised family income and hourly wages? 
for any inc hw inc55 hw55 inc25 hw25 ince hwe: gen quintile_X=int(pctile_X/20) \ recode quintile_X 0=1 1=2 2=3 3=4 4=5
for any hw hw55 hw25 hwe: replace quintile_X=0 if employed==0
tab quintile_hw quintile_inc, row nofreq
tab quintile_hw55 quintile_inc55, row nofreq
tab quintile_hw25 quintile_inc25, row nofreq
tab quintile_hw quintile_inc [aw=hours], row nofreq
tab quintile_hw55 quintile_inc55 [aw=hours], row nofreq
tab quintile_hw55 quintile_inc55 [aw=hours], row nofreq
tab quintile_hwe quintile_ince, row nofreq
gen equiv_faminc=(inctotpu-itaxpu)/(personsu^.5)
corr hwage equiv_faminc [w=wtpsn] 
gen hwage0=hwage
recode hwage0 .=0
corr hwage0 equiv_faminc [w=wtpsn] 
corr hwage equiv_faminc [w=wtpsn] if age<=55
corr hwage equiv_faminc [w=wtpsn] if emp_any==1
drop equiv_faminc

* Where are minimum wage workers on the family income distribution?
for any sub min: tab quintile_inc if X_worker==1
for any sub min: tab quintile_inc55 if X_worker==1
for any sub min: tab quintile_inc25 if X_worker==1
for any sub min: tab quintile_inc if X_worker==1 [aw=hours]
for any sub min: sum pctile_inc if X_worker==1,d \ sum pctile_inc55 if X_worker==1,d \ sum pctile_ince if X_worker==1,d

bysort pctile_inc: egen temp1=mean(min_worker)
bysort pctile_inc55: egen temp2=mean(min_worker)
bysort pctile_inc25: egen temp3=mean(min_worker)
egen tag1=tag(pctile_inc)
egen tag2=tag(pctile_inc55)
egen tag3=tag(pctile_inc25)

* How do MW workers break down across family types? 
gen famtype=.
replace famtype=1 if kidsu15>0
*replace famtype=2 if iutype==1 & kidsu15>0
*replace famtype=1 if iutype==3 & kidsu15>0
replace famtype=0 if famtype==.
tab quintile_hw famtype, col nofreq

* Over time
tabstat sub_worker min_worker if employed==1, by(yearmonth) stat(mean)

gen monthsq=month^2

* Summary statistics
for any gdp_sal mw_real: gen temp_X=exp(X)
tabstat female married educ1-educ4 age osborn employed hours hwage iwsucp mw [w=wtpsn], stats(mean sd n) col(stat)
bysort yearmonth: egen temp=median(hwage)
gen ratio=mw/temp
tabstat ratio,by(yearmonth)
drop temp ratio

tabstat female married educ1-educ4 age osborn employed hwage if employed==1 [w=wtpsn], stats(mean n) col(stat)
tabstat female married educ1-educ4 age osborn employed hwage [w=wtpsn], stats(mean n) col(stat)

replace ftptstat=ftptacp if survey==1994

save temp_all, replace

****************************************
* Distribution of MW workers
****************************************

clear
set mem 100m
cd "<Active directory>" 
use temp_all, clear

set scheme s1mono
* By age
twoway kdensity age [w=wtpsn],width(7) legend(lab(1 "Adult population")) lstyle(p1solid) || kdensity age [w=wtpsn] if employed==1,width(7) legend(lab(2 "Employed population")) lstyle(p1solid) lpattern(dash_dot) || kdensity age if sub_worker==1 [w=wtpsn],width(7) legend(lab(3 "Subminimum wage workers")) lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity age if min_worker==1 [w=wtpsn],width(7) legend(lab(4 "Minimum wage workers")) lstyle(p1solid) lpattern(dash) xtitle("Age") ytitle("Density") title("Figure 3: Age Distribution of" "Subminimum and Minimum Wage Workers", pos(6))
sum age [w=wtpsn],d

sum age if employed==1 [w=wtpsn],d
for any sub min: sum age [w=wtpsn] if X_worker==1,d

* Number of teens
gen aged1519=1 if age>=15 & age<=19
gen aged1524=1 if age>=15 & age<=24
for any aged1519 aged1524: recode X .=0 if age~=.
for any sub min: sum aged1519 aged1524 [w=wtpsn] if X_worker==1
for any sub min: sum aged1519 aged1524 [w=wtpsn] if X_worker==1 & survey==2002

* Summary statistics for MW & non-MW earners
* Adult population
tabstat female married educ1-educ4 age aged1519 aged1524 osborn employed hours hwage iwsucp mw [w=wtpsn], stats(mean sd n) col(stat)
* Employed
tabstat female married educ1-educ4 age aged1519 aged1524 osborn employed hours hwage iwsucp mw [w=wtpsn] if employed==1, stats(mean sd n) col(stat)
* Subminimum wage workers
tabstat female married educ1-educ4 age aged1519 aged1524 osborn employed hours hwage iwsucp mw [w=wtpsn] if sub_worker==1, stats(mean sd n) col(stat)
* Minimum wage workers
tabstat female married educ1-educ4 age aged1519 aged1524 osborn employed hours hwage iwsucp mw [w=wtpsn] if min_worker==1, stats(mean sd n) col(stat)
drop aged1519 aged1524

* Average number of workers in MW worker households, and rank of MW workers' hourly wages (1=highest)
bysort hhidu iuno survey: egen emp_total=count(employed)
bysort hhidu iuno survey: egen emp_rank=rank(hwage), field 
for any sub min: sum emp_total if X_worker==1  \ tab emp_total if X_worker==1
for any sub min: sum emp_rank if X_worker==1 \tab emp_rank if X_worker==1
for any sub min: sum emp_total if X_worker==1 & emp_total>1 \tab emp_rank if X_worker==1 & emp_total>1
for any sub min: sum X_worker [w=wtpsn] if employed==1

* Regressions
for any married haskids: gen female_X=female*X
for any sub min: recode X_worker .=0
gen agecu=(age^3)/10000
gen agequ=(age^4)/100000
ren educ1 stillschool 
ren educ2 unidegree
ren educ3 otherqual
global coefs "female married haskids15 female_married female_haskids stillschool unidegree otherqual age agesq agecu agequ osborn i.survey"
xi: dprobit sub_worker $coefs if employed==1, r
outreg using table_whogetsmw.doc, coefastr nocons bracket 3aster replace bdec(3) se addstat("Pseudo R-squared",e(r2_p),"Observed Prob.",e(pbar)) ct("X-Emp")
for any min: xi: dprobit X_worker $coefs if employed==1 & sub_worker~=1, r \ outreg using table_whogetsmw.doc, coefastr nocons bracket 3aster append bdec(3) se addstat("Pseudo R-squared",e(r2_p),"Observed Prob.",e(pbar)) ct("X-Emp")
for any sub: xi: dprobit X_worker $coefs if employed==1 & age>=21, r \ outreg using table_whogetsmw.doc, coefastr nocons bracket 3aster append bdec(3) se addstat("Pseudo R-squared",e(r2_p),"Observed Prob.",e(pbar)) ct("X-FullPop")
for any min: xi: dprobit X_worker $coefs if employed==1 & age>=21 & sub_worker~=1, r \ outreg using table_whogetsmw.doc, coefastr nocons bracket 3aster append bdec(3) se addstat("Pseudo R-squared",e(r2_p),"Observed Prob.",e(pbar)) ct("X-FullPop")

* By income - all
twoway kdensity pctile_inc [w=wtpsn] if sub_worker==1, lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity pctile_inc [w=wtpsn] if min_worker==1, lstyle(p1solid) lpattern(dash) xtitle("Percentile of equivalised after-tax family income") ytitle("Density") title("Sample: Aged 15+") yline(.01) name(all, replace) nodraw legend(off)
* 15-55
twoway kdensity pctile_inc55 [w=wtpsn] if sub_worker==1 & age<=55, lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity pctile_inc55 [w=wtpsn] if min_worker==1 & age<=55, lstyle(p1solid) lpattern(dash) xtitle("Percentile of equivalised after-tax family income") ytitle("Density") title("Sample: Aged 15-55") yline(.01) name(_55, replace) nodraw legend(off)
* Employed
twoway kdensity pctile_ince [w=wtpsn] if sub_worker==1,legend(lab(1 "Subminimum wage workers")) lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity pctile_ince [w=wtpsn] if min_worker==1, legend(lab(2 "Minimum wage workers")) lstyle(p1solid) lpattern(dash) xtitle("Percentile of equivalised after-tax family income") ytitle("Density") ti("Sample: Households with at least one employed person") yline(.01) name(e,replace) nodraw
gr combine all _55 e, ti("Figure 5: Family Income Distribution of" "Subminimum and Minimum Wage Workers", pos(6)) iscale(.5)

for any sub min: sum pctile_inc pctile_inc55 pctile_ince if X_worker==1,d 

* 2002-03 only
twoway kdensity pctile_inc [w=wtpsn] if sub_worker==1 & survey==2002, lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity pctile_inc [w=wtpsn] if min_worker==1 & survey==2002, lstyle(p1solid) lpattern(dash) xtitle("Percentile of equivalised after-tax family income") ytitle("Density") title("Sample: Aged 15+") yline(.01) name(all, replace) nodraw legend(off)
twoway kdensity pctile_inc55 [w=wtpsn] if sub_worker==1 & age<=55 & survey==2002, lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity pctile_inc55 [w=wtpsn] if min_worker==1 & age<=55 & survey==2002, lstyle(p1solid) lpattern(dash) xtitle("Percentile of equivalised after-tax family income") ytitle("Density") title("Sample: Aged 15-55") yline(.01) name(_55, replace) nodraw legend(off)
twoway kdensity pctile_ince [w=wtpsn] if sub_worker==1 & survey==2002,legend(lab(1 "Subminimum wage workers")) lstyle(p1solid) lpattern(dot) lwidth(medthick) || kdensity pctile_ince [w=wtpsn] if min_worker==1 & survey==2002, legend(lab(2 "Minimum wage workers")) lstyle(p1solid) lpattern(dash) xtitle("Percentile of equivalised after-tax family income") ytitle("Density") ti("Sample: Households with at least one employed person") yline(.01) name(e,replace) nodraw
gr combine all _55 e, ti("Figure 6: Family income distribution of Subminimum" "and Minimum Wage Workers (2002-03)", pos(6)) iscale(.5)

for any sub min: sum pctile_inc pctile_inc55 pctile_ince if X_worker==1 & survey==2002,d 

* Family types
for any sub_worker min_worker employed: egen temp_1=count(iutype) if X==1 \ bysort iutype: egen temp_2=count(iutype) if X==1 \ gen tX=temp_2/temp_1 \ drop temp_1 temp_2
egen temp_1=count(iutype)
bysort iutype: egen temp_2=count(iutype)
gen t=temp_2/temp_1
drop temp_1 temp_2
replace t=temployed if employed==1
replace t=tsub_worker if sub_worker==1
replace t=tmin_worker if min_worker==1
set scheme s1mono
gen workertype=1
replace workertype=2 if employed==1 
replace workertype=3 if sub_worker==1
replace workertype=4 if min_worker==1
label define workertype 1 "All adults" 2 "Employed" 3 "Subminimum wage workers" 4 "Minimum wage workers"
la values workertype workertype
label define iutype2 1 "Couple with dependents" 2 "Couple only" 3 "One parent with dependents" 4 "Single person"
label values iutype iutype2
graph bar (mean) t, over(workertype) over(iutype, label(labsize(small))) asyvars ytitle("Share") title("Figure 4: Family Types of" "Subminimum and Minimum Wage Workers", pos(6)) subtitle("Sample: Aged 15+") bar(1,fcolor(white)) bar(2,fcolor(gs14)) bar(3,fcolor(gs8)) bar(4,fcolor(black))

* Trends in minwage coverage
for any sub min: gen X_workere=X_worker if employed==1
collapse min_worker* sub_worker* [w=wtpsn],by(survey)
gen date=survey
la var sub_workere "Subminimum wage workers"
la var min_workere "Minimum wage workers"
twoway line sub_workere date,lstyle(p1solid) lpattern(dot) lwidth(medthick) || line min_workere date , lstyle(p1solid) lpattern(dash) ysc(range(0 .14)) ylabel(0(.02).14) xtitle("") ytitle("Share of employees") ti("Figure 2: Proportion of Employees" "Earning Subminimum and Minimum Wages", pos(6))
tabstat sub_workere min_workere,by(date)


*********************************
* Simulated MW rise
* dearn 1-3 correspond with simulations 1-3 in Table 4 of the paper
* dearn 4 allows for testing of the range for equality promotion, by changing the 
* hourly wage and labour demand elasticities (ehw & eld)
*********************************

global ehw .5
global eld -.1
forvalues i=1/50 {
clear
set mem 100m
cd "<Active directory>" 
use temp_all if survey==2002, clear
recode iwsucp 1=0
recode inctotcu 1=0
gen random=uniform()
* Effects on weekly earnings
gen dearn1=iwsucp*.1 if sub_worker==1 | min_worker==1 
gen dearn2=-iwsucp+.001 if (sub_worker==1 | min_worker==1) & random<=.1
gen dearn3=dearn1
gen dearn4=dearn1*$ehw
replace dearn3=-iwsucp+.001 if (sub_worker==1 | min_worker==1) & random<=.1
replace dearn4=-iwsucp+.001 if (sub_worker==1 | min_worker==1) & random<=.1*-$eld
for num 1/4: recode dearnX .=0
gen oldearn=iwsucp
for num 1/4: gen newearnX=oldearn+dearnX
* Effects on hourly wages
gen dearn1h=hwage*.1 if sub_worker==1 | min_worker==1 
gen dearn2h=-hwage if (sub_worker==1 | min_worker==1) & random<=.1
gen dearn3h=dearn1h
gen dearn4h=dearn1h*$ehw
replace dearn3h=-hwage if (sub_worker==1 | min_worker==1) & random<=.1
replace dearn4h=-hwage if (sub_worker==1 | min_worker==1) & random<=.1*-$eld
for num 1/4: recode dearnXh .=0
gen oldhw=hwage
for num 1/4: gen newhwX=oldhw+dearnXh
* Recoding dismissals as missing hourly wages
for num 2/4: replace newhwX=. if dearnXh==-hwage
* Summing weekly earnings effects to the household level
for num 1/4: bysort hhidu iuno survey: egen dearnX_iu=sum(dearnX)
gen oldinc=(inctotcu)/(personsu^.5)
for num 1/4: gen newincX=(inctotcu+dearnX_iu)/(personsu^.5)
for var oldinc newinc*: recode X min/0=0

* Calculating inequality
drop temp*
foreach X of varlist oldinc newinc* {
  ineqdeco `X' [aw=wtunit] if `X'>0
  gen _9010`X' = $S_9010 
  gen _gini`X' = $S_gini
  gen _a1`X' = $S_a1
  gen _a2`X' = $S_a2
  sum `X' [aw=wtunit] if `X'>0,d
  gen temp1=r(p50)
  gen temp2=1 if `X'<temp1/2 & `X'>0
  replace temp2=0 if temp2==. & `X'>0 & `X'~=.
  egen _pov`X'=mean(temp2)
  drop temp*
  }
foreach X of varlist oldearn newearn* oldhw newhw* {
  ineqdeco `X' [aw=wtpsn] if `X'>0
  gen _9010`X' = $S_9010 
  gen _gini`X' = $S_gini
  gen _a1`X' = $S_a1
  gen _a2`X' = $S_a2
  sum `X' [aw=wtunit] if `X'>0,d
  gen temp1=r(p50)
  gen temp2=1 if `X'<temp1/2 & `X'>0
  replace temp2=0 if temp2==. & `X'>0 & `X'~=.
  egen _pov`X'=mean(temp2)
  drop temp*
  }
sum _9* _gini* _a1* _a2* _pov*, separator(0)
gen n=_n
keep if n==`i'
sort n
save temp_sim`i', replace
}

* Aggregating the 50 simulations
use temp_sim1, clear
for num 2/50: sort n \ merge n using temp_simX \ drop _merge
collapse _9* _gini* _a1* _a2* _pov*
sum _9* _gini* _a1* _a2* _pov*, separator(0)
save temp_sim_agg, replace
for num 1/50: erase temp_simX.dta
sum _gini* _a1* _pov*, separator(0)

for var oldinc newinc*: inequal2 X [w=wtunit] if age>=25 & age<=55
for var oldinc newinc*: inequal2 X [w=wtunit] if employed==1

*********************************
* Graphing Federal Minimum Wage
*********************************
cd "<Active directory>" 
clear
set more off
set obs 12
gen year=1993
gen n=_n
replace year=year+n-1
expand 12
bysort year: gen nn=_n
gen date = mdy(nn,1,year)
format date %dm_CY
gen yearmonth=year*100+nn
gen mw_fed=0
for num 325.4 333.4 333.4 341.4 349.4 359.4 373.4 385.4 400.4 413.4 431.4 448.4 467.4 \ num 199109 199312 199411 199510 199605 199705 199805 199905 200005 200105 200205 200305 200405: replace mw_fed=X/38 if yearmonth>=Y 
gen cpi=.
for num 199402 199405 199408 199411 199502 199505 199508 199511 199602 199605 199608 199611 199702 199705 199708 199711 199802 199805 199808 199811 199902 199905 199908 199911 200002 200005 200008 200011 200102 200105 200108 200111 200202 200205 200208 200211 200302 200305 200308 200311 200402 200405 200408 200411 \ num 114.60 115.30 116.10 116.70 117.80 119.00 120.50 121.50 121.90 122.80 123.30 124.20 125.10 125.40 125.20 126.10 126.50 127.20 127.52 128.15 128.04 128.57 129.68 130.26 131.22 132.12 133.15 133.56 134.98 136.10 136.51 137.73 138.95 139.96 140.88 141.90 143.73 143.73 144.54 145.25 146.58 147.29 147.90 149.02: replace cpi=Y/147.9 if yearmonth>=X
la var mw_fed "Current dollars"
gen mw_fed_real=mw_fed/cpi
la var mw_fed_real "2004 dollars"
sort yearmonth
drop if date<12570
set scheme s1mono
twoway line mw_fed date,lstyle(p1solid) || line mw_fed_real date, lstyle(p1solid) lpattern(dash) ytitle("Hourly minimum wage") ti("Figure 1: Federal Minimum Wage, 1994-2004", pos(6)) xtitle("") note("1994-1996: C14 Metal Industry Award under Accord VII. 1997-2004: Federal Minimum Wage.",span)
for var mw*: gen temp2=X if yearmonth==199406 \ egen temp3=max(temp2) \ gen temp5=X if yearmonth==200408 \ egen temp6=max(temp5) \ gen dX=temp6/temp3 \ drop temp*
* Note: comparison is from mid-1994 to mid-2004
sum mw* if yearmonth==199406 
sum mw* if yearmonth==200408 
sum dmw*

